/*************************************************
*     rpld - an IBM style RIPL server            *
*************************************************/

/* Copyright (c) 1999,2000, James McKenzie.
 *                      All rights reserved
 * Copyright (c) 1998,2000, Christopher Lightfoot.
 *                      All rights reserved
 *
 * By using this file, you agree to the terms and conditions set
 * forth in the LICENCE file which can be found at the top level of
 * the rpld distribution.
 *
 * IBM is a trademark of IBM corp.
 *
 */



static char rcsid[] = "$Id: rpl.c,v 1.10 2000/07/17 10:43:34 root Exp $";

/*
 * $Log: rpl.c,v $
 * Revision 1.10  2000/07/17 10:43:34  root
 * #
 *
 * Revision 1.9  2000/07/16 14:05:28  root
 * #
 *
 * Revision 1.8  2000/07/16 13:18:10  root
 * #
 *
 * Revision 1.1  2000/07/16 13:16:33  root
 * #
 *
 * Revision 1.7  1999/09/13 11:17:35  root
 * \#
 *
 * Revision 1.6  1999/09/13 11:08:34  root
 * \#
 *
 * Revision 1.5  1999/09/13 11:05:27  root
 * \#
 *
 * Revision 1.4  1999/09/13 11:04:13  root
 * \#
 *
 */

#include "project.h"

/* Code for recursively disembling packets */

static void
rpl_token (struct rpl_packet *p, int s, unsigned char *pp, int pl)
{
  int i;
  switch (s)
    {
    case RPL_TK_TMZ:
      if (pl < 4)
        return;
      p->flags |= RPL_FL_TMZ;
      p->themightyzero = ntohl (*(u32 *) pp);
      return;
    case RPL_TK_MYMAC:
      if (pl < ETH_ALEN)
        return;
      p->flags |= RPL_FL_MYMAC;
      bcopy (pp, p->mymac, ETH_ALEN);
      return;
    case RPL_TK_SAP:
      /*FIXME: */
      p->flags |= RPL_FL_SAP;
      return;
    case RPL_TK_FRAMELEN:
      if (pl < 2)
        return;
      p->flags |= RPL_FL_FRAMELEN;
      p->framelen = ntohs (*(u16 *) pp);
      return;
    case RPL_TK_WHOAMI:
      if (pl < 2)
        return;
      p->flags |= RPL_FL_WHOAMI;
      p->whoami = ntohs (*(u16 *) pp);
      return;
    case RPL_TK_TSZ:
      if (pl < 1)
        return;
      p->flags |= RPL_FL_TSZ;
      p->thesmallzero = *pp;
      return;
    case RPL_TK_YOUMAC:
      if (pl < ETH_ALEN)
        return;
      p->flags |= RPL_FL_YOUMAC;
      bcopy (pp, p->youmac, ETH_ALEN);
      return;
    case RPL_TK_BLOCK:
      if (pl < 4)
        return;
      p->flags |= RPL_FL_BLOCK;
      p->block = ntohl (*(u32 *) pp);
      return;
    case RPL_TK_DATA:
      if (NOTINRANGE (1, pl, MAX_DATA_LEN))
        return;
      p->flags |= RPL_FL_DATA;
      bcopy (pp, p->data, pl);
      p->datalen = pl;
      return;
    case RPL_TK_IDENT:
      if (NOTINRANGE (1, pl, MAX_IDENT_LEN))
        return;
      p->flags |= RPL_FL_IDENT;
      bcopy (pp, p->ident, pl);
      p->identlen = pl;
      return;
    case RPL_FL_ADDR:
      if (pl < 9)
        return;
      p->flags |= RPL_FL_ADDR;
      p->addr.load = ntohl (*(u32 *) pp);
      p->addr.run = ntohl (*(u32 *) (pp + 4));
      p->addr.flags = pp[8];
      return;
    default:
      syslog (LOG_ERR, "Unknown token: 0x%04x", s);
    }

}

static void
rpl_frag (struct rpl_packet *p, unsigned char *pp, int pl)
{
  int s;

  while (pl > 0)
    {

      if (pl < 2)
        return;
      s = ntohs (*(u16 *) pp);

      if (!s)
        {
          syslog (LOG_ERR, "Unexpected end of packet", s);
          return;

        }
      if (RPL_IS_TOKEN (s))
        {
          rpl_token (p, s, pp + 2, pl - 2);
          return;
        }
      else
        {
          rpl_frag (p, pp + 2, s - 2);
          pp += s;
          pl -= s;
        }

    }

}

/* Code for assembling packets */
int
write_char (unsigned char *ptr, u8 v)
{
  *ptr = v;
  return (1);
}

int
write_short (unsigned char *ptr, u16 v)
{
  *((u16 *) ptr) = htons (v);
  return (2);
}

int
write_long (unsigned char *ptr, u32 v)
{
  *((u32 *) ptr) = htonl (v);
  return (4);
}

int
put_char (unsigned char *ptr, u16 token, u8 value)
{
  int len = 0, i;
  i = write_short (ptr, 0x2 + 0x2 + 0x1);
  ptr += i;
  len += i;
  i = write_short (ptr, token);
  ptr += i;
  len += i;
  i = write_char (ptr, value);
  ptr += i;
  len += i;
  return (len);
}

int
put_short (unsigned char *ptr, u16 token, u16 value)
{
  int len = 0, i;
  i = write_short (ptr, 0x2 + 0x2 + 0x2);
  ptr += i;
  len += i;
  i = write_short (ptr, token);
  ptr += i;
  len += i;
  i = write_short (ptr, value);
  ptr += i;
  len += i;
  return (len);
}

int
put_long (unsigned char *ptr, u16 token, u32 value)
{
  int len = 0, i;
  i = write_short (ptr, 0x2 + 0x2 + 0x4);
  ptr += i;
  len += i;
  i = write_short (ptr, token);
  ptr += i;
  len += i;
  i = write_long (ptr, value);
  ptr += i;
  len += i;
  return (len);
}

int
put_mac (unsigned char *ptr, u16 token, unsigned char *mac)
{
  int len = 0, i;
  i = write_short (ptr, 0x2 + 0x2 + 0x6);
  ptr += i;
  len += i;
  i = write_short (ptr, token);
  ptr += i;
  len += i;
  bcopy (mac, ptr, ETH_ALEN);
  ptr += ETH_ALEN;
  len += ETH_ALEN;
  return (len);
}

static void
send_found_frame (struct nit *n, unsigned char *dmac, struct rpl_packet *out)
{
  unsigned char buf[MAX_FRAME_LEN], *ptr;
  int len, i;

  if (out->flags != RPL_FOUND_FLAGS)
    {

      syslog (LOG_ERR, "FOUND packet: doesn't match RE case %x!=%x",
              out->flags, RPL_FOUND_FLAGS);
      return;
    }

  ptr = buf;
  len = 0;

  i = write_short (ptr, 0x0);
  ptr += i;
  len += i;                     /*length */
  i = write_short (ptr, out->type);
  ptr += i;
  len += i;                     /*type */

  i = put_long (ptr, RPL_TK_TMZ, out->themightyzero);
  ptr += i;
  len += i;
  i = put_char (ptr, RPL_TK_TSZ, out->thesmallzero);
  ptr += i;
  len += i;
  i = put_mac (ptr, RPL_TK_YOUMAC, out->youmac);
  ptr += i;
  len += i;
  i = put_mac (ptr, RPL_TK_MYMAC, out->mymac);
  ptr += i;
  len += i;
  i = write_short (ptr, 0x10);
  ptr += i;
  len += i;
  i = write_short (ptr, 0x08);
  ptr += i;
  len += i;
  i = put_short (ptr, RPL_TK_FRAMELEN, out->framelen);
  ptr += i;
  len += i;
  i = put_short (ptr, RPL_TK_WHOAMI, out->whoami);
  ptr += i;
  len += i;
  i = put_char (ptr, RPL_TK_SAP, out->sap);
  ptr += i;
  len += i;

  ptr = buf;
  write_short (ptr, len);       /*length */

  send_llc_frame (n, RPL_SAP, RPL_SAP, dmac, buf, len);

}


static void
send_file_data_frame (struct nit *n, unsigned char *dmac,
                      struct rpl_packet *out)
{
  unsigned char buf[MAX_FRAME_LEN], *ptr;
  int len, i;

  if (out->flags != RPL_FILE_DATA_FLAGS)
    {
      syslog (LOG_ERR, "FILE.DATA packet: doesn't match RE case %x!=%x",
              out->flags, RPL_FILE_DATA_FLAGS);
      return;
    }

  ptr = buf;
  len = 0;

  i = write_short (ptr, 0x0);
  ptr += i;
  len += i;                     /*length */
  i = write_short (ptr, out->type);
  ptr += i;
  len += i;                     /*type */

  i = put_long (ptr, RPL_TK_BLOCK, out->block);
  ptr += i;
  len += i;

  i = write_short (ptr, 0x2 + 0x2 + 0x9);
  ptr += i;
  len += i;
  i = write_short (ptr, RPL_TK_ADDR);
  ptr += i;
  len += i;
  i = write_long (ptr, out->addr.load);
  ptr += i;
  len += i;
  i = write_long (ptr, out->addr.run);
  ptr += i;
  len += i;
  i = write_char (ptr, out->addr.flags);
  ptr += i;
  len += i;

  i = write_short (ptr, 0x2 + 0x2 + out->datalen);
  ptr += i;
  len += i;
  i = write_short (ptr, RPL_TK_DATA);
  ptr += i;
  len += i;
  bcopy (out->data, ptr, out->datalen);
  ptr += out->datalen;
  len += out->datalen;


  ptr = buf;
  write_short (ptr, len);       /*length */

  send_llc_frame (n, RPL_SAP, RPL_SAP, dmac, buf, len);


}

void
rpl_send_packet (struct nit *n, unsigned char *d, struct rpl_packet *p)
{

  switch (p->type)
    {
    case RPL_PK_FOUND:
      send_found_frame (n, d, p);
      break;
    case RPL_PK_FILEDATA:
      send_file_data_frame (n, d, p);
      break;
    default:
      syslog (LOG_ERR, "rpl_send_packet: unknown packet type 0x%04x",
              p->type);
    }
}

/* The llc code calls this whenever it gets a valid rpl packet */
void
rpl_packet_recvd (struct nit *n, unsigned char *pptr, int plen)
{

  struct rpl_packet p;
  int elen;

  if (plen < 6)
    return;

  elen = ntohs (*(u16 *) pptr);

  if (elen != plen)
    {
      syslog (LOG_ERR, "received lengths disagree %d vs %d", elen, plen);
    }

  if (elen > plen)
    elen = plen;

  pptr += 2;                    /*Length */
  elen -= 2;

  if (elen < 2)
    return;
  p.type = ntohs (*(u16 *) pptr);

  pptr += 2;
  elen -= 2;

  p.flags = 0;

  rpl_frag (&p, pptr, elen);    /*Analyse the packet */

  rpl_packet_recvd_callback (n, &p); /*Pass it up to the protocol layer */

}
